#!/bin/bash

# 系统优化配置脚本
# 功能：根据内存大小自动配置交换空间，并设置其他系统参数

# 检查是否为root用户
if [ "$(id -u)" -ne 0 ]; then
    echo "Please execute this script using the root user!"
    exit 1
fi

# 获取系统总内存(GB)
TOTAL_MEM=$(free -g | awk '/Mem:/ {print $2}')
SWAP_FILE="/swapfile"
SWAP_FILE_IMG="/swap.img"

echo "Current system memory: ${TOTAL_MEM}GB"

# 1. 交换空间配置
if [ "$TOTAL_MEM" -lt 16 ]; then
    # 内存小于16GB，检查是否已有足够的交换空间
    CURRENT_SWAP_MB=$(free -m | awk '/Swap:/ {print $2}')
    
    if [ "$CURRENT_SWAP_MB" -ge 3800 ]; then
        echo "It has been detected that the current swap space of ${CURRENT_SWAP_MB}MB is greater than or equal to 4GB, so no adjustment is necessary."
    else
        echo "It was detected that the memory is less than 16GB and the swap space is less than 4GB (currently ${CURRENT_SWAP_MB}MB). A 4GB swap space will be configured"
        
        # 如果已有交换文件，先禁用
        if swapon --show | grep -q "$SWAP_FILE"; then
            swapoff "$SWAP_FILE"
            rm -f "$SWAP_FILE"
        fi
        
        # 创建交换文件
        if ! fallocate -l 4G "$SWAP_FILE"; then
            echo "fallocate operation failed. Attempting to create the swap file using dd"
            dd if=/dev/zero of="$SWAP_FILE" bs=1M count=4096
        fi
        
        chmod 600 "$SWAP_FILE"
        mkswap "$SWAP_FILE"
        swapon "$SWAP_FILE"
        
        # 永久生效
        if ! grep -q "$SWAP_FILE" /etc/fstab; then
            echo "$SWAP_FILE none swap sw 0 0" >> /etc/fstab
        fi
        
        # 调整swappiness
        if ! grep -q "vm.swappiness" /etc/sysctl.conf; then
            echo "vm.swappiness=30" >> /etc/sysctl.conf
        else
            sed -i 's/^vm.swappiness=.*/vm.swappiness=30/' /etc/sysctl.conf
        fi
    fi
else
    # 内存大于等于16GB，禁用交换空间
    echo "It has been detected that the memory is greater than or equal to 16GB. Swap space is disabled."
    
    # 禁用所有交换空间
    swapoff -a
    
    # 从fstab中注释掉交换空间行
    sed -i '/swap/s/^/#/' /etc/fstab
    
    # 删除交换文件（如果存在）
    if [ -f "$SWAP_FILE" ]; then
        rm -f "$SWAP_FILE"
    fi
    if [ -f "$SWAP_FILE_IMG" ]; then
        rm -f "$SWAP_FILE_IMG"
    fi

fi

# 2. 关闭自动升级
echo "Disable the system automatic upgrade service"
systemctl stop unattended-upgrades
systemctl disable unattended-upgrades

# 3. 文件句柄限制配置
echo "Optimization of system file handle count"

# 修改limits.conf
if ! grep -q "* soft nofile 1048576" /etc/security/limits.conf; then
    echo "* soft nofile 1048576" >> /etc/security/limits.conf
    echo "* hard nofile 1048576" >> /etc/security/limits.conf
    echo "root soft nofile 1048576" >> /etc/security/limits.conf
    echo "root hard nofile 1048576" >> /etc/security/limits.conf
fi

echo 'vm.overcommit_memory = 1' >> /etc/sysctl.conf

# 系统全局文件句柄数
echo "fs.file-max = 2097152" >> /etc/sysctl.conf
echo "fs.nr_open = 2098576" >> /etc/sysctl.conf
echo 2097152 > /proc/sys/fs/nr_open

# systemd服务文件句柄限制
sed -i 's/^#DefaultLimitNOFILE=/DefaultLimitNOFILE=1048576/' /etc/systemd/system.conf

# 4. 网络优化配置
echo "System network parameter optimization"

# 并发连接设置
cat >> /etc/sysctl.conf <<EOF
net.core.somaxconn=32768
net.ipv4.tcp_max_syn_backlog=16384
net.core.netdev_max_backlog=16384
EOF

# TCP Socket Buffer设置
cat >> /etc/sysctl.conf <<EOF
net.core.rmem_default=262144
net.core.wmem_default=262144
net.core.rmem_max=16777216
net.core.wmem_max=16777216
net.core.optmem_max=16777216
net.ipv4.tcp_rmem=1024 4096 16777216
net.ipv4.tcp_wmem=1024 4096 16777216
EOF

# TCP连接追踪设置
# 仅当nf_conntrack模块存在时才设置相关参数
if [ -f /proc/sys/net/netfilter/nf_conntrack_max ]; then
    cat >> /etc/sysctl.conf <<EOF
net.netfilter.nf_conntrack_max=1000000
net.netfilter.nf_conntrack_tcp_timeout_time_wait=30
EOF
elif [ -f /proc/sys/net/nf_conntrack_max ]; then
    cat >> /etc/sysctl.conf <<EOF
net.nf_conntrack_max=1000000
EOF
fi

# 5. 应用所有sysctl设置
sysctl -p

# 6. 设置locale
update-locale LANG=C.UTF-8

# 7. 设置当前会话的文件句柄限制
ulimit -n 1048576

echo "System optimization configuration completed!"

# 显示当前配置状态
echo -e "\nCurrent system configuration status："
echo "Memory and swap space:"
free -h
echo -e "\nSwap space status:"
swapon --show
echo -e "\nFile handle limit:"
ulimit -n
echo -e "\nnetwork parameter:"
sysctl net.core.somaxconn net.ipv4.tcp_max_syn_backlog net.core.netdev_max_backlog