#!/usr/bin/env python3
import zipfile
import os
import tempfile
import shutil
import sys

def read_env_config(conf_file):
    """从env.conf文件读取配置"""
    config = {}
    with open(conf_file, 'r', encoding='utf-8') as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith('#'):
                continue
            if '=' in line:
                key, value = line.split('=', 1)
                config[key.strip()] = value.strip()
    return config


def minimal_jar_update(jar_path, config):
    
    # backup_path = jar_path + '.backup'
    # shutil.copy2(jar_path, backup_path)
    
    try:
        temp_jar = jar_path + '.temp'
        
        with zipfile.ZipFile(jar_path, 'r') as zip_in, \
             zipfile.ZipFile(temp_jar, 'w') as zip_out:
            
            for item in zip_in.infolist():
                data = zip_in.read(item.filename)
                
                if item.filename == 'BOOT-INF/classes/application-druid.yml':
                    content = data.decode('utf-8')
                    # 精确替换druid配置
                    content = update_yaml_value(content, 'username', config.get("spring.datasource.druid.username", ""))
                    content = update_yaml_value(content, 'password', config.get("spring.datasource.druid.password", ""))
                    data = content.encode('utf-8')
                    
                elif item.filename == 'BOOT-INF/classes/application.yml':
                    content = data.decode('utf-8')
                    # 精确替换application配置
                    content = update_yaml_value(content, 'super-secret', config.get("mqtt.super-secret", ""))
                    content = update_yaml_value(content, 'api-key', config.get("mqtt.api-key", ""))
                    content = update_yaml_value(content, 'api-secret', config.get("mqtt.api-secret", ""))
                    content = update_yaml_value(content, 'org', config.get("influx.org", ""))
                    content = update_yaml_value(content, 'bucket', config.get("influx.bucket", ""))
                    content = update_yaml_value(content, 'token', config.get("influx.token", ""))
                    content = update_yaml_value(content, 'secret', config.get("token.secret", ""))
                    data = content.encode('utf-8')
                
                zip_out.writestr(item, data)
        
        shutil.move(temp_jar, jar_path)
        return True
        
    except Exception as e:
        print(f"Application configuration failed: {e}")
        # if os.path.exists(backup_path):
        #     shutil.move(backup_path, jar_path)
        return False

def update_yaml_value(yaml_content, key, new_value):
    lines = yaml_content.split('\n')
    new_lines = []
    
    for line in lines:
        stripped = line.strip()
        # 检查是否以目标键开头（后面跟着冒号）
        if stripped.startswith(f"{key}:") or stripped.startswith(f"{key} :"):
            # 检查该行是否已经有值（冒号后面有非空内容）
            if ':' in line:
                parts = line.split(':', 1)
                existing_value = parts[1].strip()
                
                # 只有当该键已经有值时才进行替换
                if existing_value:
                    # 保持原始缩进，只替换值部分
                    indent = line[:len(line) - len(line.lstrip())]
                    # 如果值包含空格或特殊字符，添加引号
                    if ' ' in str(new_value) or any(char in str(new_value) for char in ['"', "'", ':', '#']):
                        new_line = f'{indent}{key}: "{new_value}"'
                    else:
                        new_line = f'{indent}{key}: {new_value}'
                    new_lines.append(new_line)
                else:
                    # 如果键没有值，保持原样
                    new_lines.append(line)
            else:
                # 如果没有冒号（理论上不应该发生），保持原样
                new_lines.append(line)
        else:
            new_lines.append(line)
    
    return '\n'.join(new_lines)

def update_jar(jar_directory, jar_filename, config):
    jar_file = os.path.join(jar_directory, jar_filename)
    # 检查JAR文件是否存在
    if not os.path.exists(jar_file):
        print(f"Unable to locate the application package: {jar_file}")
        sys.exit(1)
    
    # 执行更新
    success = minimal_jar_update(jar_file, config)
    if success:
        print(f"{jar_filename} configuration successful")
    else:
        print(f"{jar_filename} configuration failed!")
        sys.exit(1)

def main():
    # 配置文件路径
    conf_file = '/data/init/build_env.conf'
    
    # 从env.conf读取配置
    print(f"Read configuration from {conf_file}")
    config = read_env_config(conf_file)
    
    safe_config = {}
    for key, value in config.items():
        safe_config[key] = value

    update_jar("/data/auth", "dhcs-emqx.jar", safe_config)
    update_jar("/data/dhcs", "dhcs-portal.jar", safe_config)

if __name__ == "__main__":
    main()