local _M = {}

-- 固定映射表
local original_mapping = {
    ["0"] = "a7", ["1"] = "b9", ["2"] = "c3", ["3"] = "d5", ["4"] = "e8", ["5"] = "f2",
    ["6"] = "g4", ["7"] = "h6", ["8"] = "i1", ["9"] = "j0", A = "kZ", B = "lY", C = "mX",
    D = "nW", E = "oV", F = "pU", G = "qT", H = "rS", I = "sR", J = "tQ", K = "uP",
    L = "vO", M = "wN", N = "xM", O = "yL", P = "zK", Q = "A0", R = "B1", S = "C2",
    T = "D3", U = "E4", V = "F5", W = "G6", X = "H7", Y = "I8", Z = "J9", a = "Ka",
    b = "Lb", c = "Mc", d = "Nd", e = "Oe", f = "Pf", g = "Qg", h = "Rh", i = "Si",
    j = "Tj", k = "Uk", l = "Vl", m = "Wm", n = "Xn", o = "Yo", p = "Zp", q = "0q",
    r = "1r", s = "2s", t = "3t", u = "4u", v = "5v", w = "6w", x = "7x", y = "8y",
    z = "9z", ["."] = "aA", _ = "bB", ["-"] = "cC"
}

-- 获取所有键
local keys = {}
for k, _ in pairs(original_mapping) do
    table.insert(keys, k)
end
table.sort(keys)  -- 确保顺序一致

-- 共享内存缓存
local cache = ngx.shared.mixture_cache

--- 解码单个字符
local function decode_char(encoded, reversed_mapping)
    -- 优先尝试2字符解码
    if #encoded >= 2 then
        local decoded = reversed_mapping[encoded:sub(1, 2)]
        if decoded then
            return decoded, 2
        end
    end
    -- 无法解码则原样返回
    return encoded:sub(1, 1), 1
end

-- 获取偏移后的映射表
local function get_shifted_mapping(shift)
    local shifted_mapping = {}
    local total_keys = #keys
    for i = 1, total_keys do
        local original_key = keys[i]
        local shifted_index = (i - 1 + shift) % total_keys + 1
        shifted_mapping[original_key] = original_mapping[keys[shifted_index]]
    end
    return shifted_mapping
end

-- 获取反转的偏移映射表
local function get_reversed_shifted_mapping(shift)
    local shifted_mapping = get_shifted_mapping(shift)
    local reversed_mapping = {}
    for k, v in pairs(shifted_mapping) do
        reversed_mapping[v] = k
    end
    return reversed_mapping
end

--- 主解码函数
function _M.mixture_decode(text)
    if #text < 2 then return text end

    -- 提取偏移量（最后2个字符）
    local encoded_shift = text:sub(-2)
    local reverse_original_mapping = {}
    for k, v in pairs(original_mapping) do
        reverse_original_mapping[v] = k
    end
    local shift_str = reverse_original_mapping[encoded_shift] or encoded_shift
    local shift = tonumber(shift_str) or 0
    shift = math.max(0, math.min(9, shift))

    -- 生成反向映射表
    local reversed_mapping = {}
    local shifted_mapping = get_shifted_mapping(shift)
    for k, v in pairs(shifted_mapping) do
        reversed_mapping[v] = k
    end

    -- 解码内容（跳过最后2个偏移字符）
    local encoded_content = text:sub(1, -3)
    local decoded = ""
    local i = 1
    while i <= #encoded_content do
        local char, consumed = decode_char(encoded_content:sub(i), reversed_mapping)
        decoded = decoded .. char
        i = i + consumed
    end

    return decoded
end

--- 带缓存的解码
local function cached_decode(text)
    local cached = cache:get(text)
    if cached then return cached end

    local decoded = _M.mixture_decode(text)
    if decoded ~= text then  -- 只有成功解码才缓存
        cache:set(text, decoded, 60)
    end
    return decoded
end

-- 动态设置 sub_filter 规则
local function setup_sub_filter(en_host, en_port)
    local web_path = "/web-ops/" .. en_host .. "/" .. en_port .. "/"
    ngx.ctx.sub_filter_rules = {
        -- 基本路径替换
        { 'href="/',      'href="' .. web_path .. '/' },
        { 'src="/',       'src="' .. web_path .. '/' },
        { 'url\\(/',      'url(' .. web_path .. '/' },
        { "url\\('/",     "url('" .. web_path .. '/' },
        -- 相对路径处理
        { 'href="./',     'href="' .. web_path .. '/' },
        { 'src="./',      'src="' .. web_path .. '/' },
        { 'href="../',    'href="' .. web_path .. '/../' },
        { 'src="../',     'src="' .. web_path .. '/../' },
        -- JavaScript location.href 处理
        { 'location.href="',    'location.href="' .. web_path .. '/" + "' },
        { "location.href='",    "location.href='" .. web_path .. "/' + '" },
        { 'location.href = "',  'location.href = "' .. web_path .. '/" + "' },
        { "location.href = '",  "location.href = '" .. web_path .. "/' + '" }
    }
    ngx.log(ngx.INFO, "sub filter process complted")
end

-- 请求处理入口
function _M.handle_request()
    local uri = ngx.var.request_uri
    local en_host, en_port, proxy_req_uri = uri:match("^/web%-ops/([^/]+)/([^/]+)(.*)")
    --ngx.log(ngx.INFO, "En_Host:", en_host, " PEn_port:", en_port, " Req_uri:", proxy_req_uri)
    if not en_host or not en_port then
        ngx.log(ngx.ERR, "Invalid URL format: ", uri)
        return ngx.exit(404)
    end

    ngx.var.en_host = en_host
    ngx.var.en_port = en_port
    ngx.var.proxy_req_uri = proxy_req_uri

    -- 解码参数并设置缓存
    local host_ok, decoded_host = pcall(function()
         return cached_decode(en_host)
    end)

    local port_ok, decoded_port = pcall(function()
         return cached_decode(en_port)
    end)

    if not host_ok or not port_ok then
        ngx.log(ngx.ERR, "Decode failed! host_ok is: ", host_ok, " port_ok is: ", port_ok)
        return ngx.exit(404)
    end

    --ngx.log(ngx.INFO, "D_Host:", decoded_host, ", D_Port:", decoded_port)
    ngx.var.decoded_host = decoded_host
    ngx.var.decoded_port = decoded_port

    -- 设置动态替换规则
    --setup_sub_filter(en_host, en_port)
end

-- 响应体过滤
function _M.body_filter()
    if not ngx.ctx.sub_filter_rules or ngx.arg[2] ~= true then
        return  -- 非最终chunk或无替换规则时跳过
    end

    local chunk = ngx.arg[1]
    if chunk then
        for _, rule in ipairs(ngx.ctx.sub_filter_rules) do
            chunk = chunk:gsub(rule[1], rule[2])
        end
        ngx.arg[1] = chunk
    end
end

return _M